/*
 * Quark2InDesign.jsx
 * Javascript for InDesign CS2 and CS3
 *
 * PURPOSE:
 * ========
 * Convert QuarkXPress files contained in folder structures to InDesign
 *
 * Walk a folder structure and look for QuarkXPress documents to convert.
 * If found one, open it and save as InDesign document.
 * The main purpose of this script is to convert QuarkXPress documents contained
 * in folders - like archives - to create ePaper XML usable for web applications.
 * The actual transform to epaper would be made by "BatchXSLT for InDEsign" (see www.aiedv.ch).
 * After having converted the Quark files to InDesign, the script 'BXSLTExport.jsx'
 * can be started to control the automated creation of epaper XML.
 *
 * IMPORTANT NOTES
 * ===============
 * 1. MAKE SURE, INDESIGN HAS A LOT OF MEMORY AVAILABLE!
 *    Around 1GB... or it may crash.
 * 2. INSTALL ALL FONTS USED IN THE DOCUMENTS!
 *    Otherwise the InDesign documents will flow with replaced fonts.
 *
 * DISCLAIMER:
 * ===============
 * Absolutely no warranty. Use it as is or modify it to match your needs
 *
 * Author: Andreas Imhof, www.aiedv.ch
 * Date: 20070414
 */

// ****** Configurable SETTINGS ******
var silent = false;									// true to show NO completion message(s), false otherwise
var overwriteExistingINDD = false;					// true to always recreate Indesing documents
													// false to check if INDD exists. if YES continue with next

var convertFolder = "";								// a default target path to search for Quark documents
//var convertFolder = "~/Export/in";				// set it to any path which should act as default export folder: no folder choose dialog will be shown
													// set to "" to show a dialog to choose the folder to convert

// ****** END OF Configurable SETTINGS ******

var numDocsConverted = 0;


main();


/****************************
 * here we go.....
 */
function main() {
	// if a document is open, inform user to close it
	if (app.documents.length != 0) {
		alert("Please, close all open documents and then restart this Script!");
		return;
	}

	// no other document is open - let's go
	var baseConvertFolder = null;
	if (convertFolder == "") {	// if no default path given - get one from user
		baseConvertFolder = Folder.selectDialog ("Choose a Folder to Convert");
		if (baseConvertFolder == null) {
			if (!silent) alert("You decided to make no Conversions.");
			return;
		}
	}
	else baseConvertFolder = new Folder(convertFolder);

	if (baseConvertFolder.exists == false) {
		alert("Conversion Folder does NOT exist: \n'" + baseConvertFolder.fsName + "'\n\nExport aborted!");
		return;
	}

	// change some settings
	var scrpref = app.scriptPreferences;
	scrpref.userInteractionLevel = UserInteractionLevels.neverInteract;
	var old_display_settings = app.displayPerformancePreferences.defaultDisplaySettings
	app.displayPerformancePreferences.defaultDisplaySettings = ViewDisplaySettings.highQuality;

	// let's do the conversion
	walkFolder(baseConvertFolder);	// call the folder walker and exporter

	// restore settings
	app.displayPerformancePreferences.defaultDisplaySettings = old_display_settings;
	scrpref.userInteractionLevel = UserInteractionLevels.interactWithAll;
	$.gc();

	// we are done...
	if (!silent) {
		var message = "QuarkXPress to InDesign Conversion complete.\n" + numDocsConverted + " QuarkXPress documents converted to InDesign.\n\nFolder:\n"+ baseConvertFolder.fsName;
		alert(message);
	}
}



/****************************
 * walk a folder tree and convert all qxp files to indd
 */
function walkFolder(theFolder) {	// walk a folder object
	if(theFolder == null) return;
	var contentArr = theFolder.getFiles();	// get all contained files and folders
	for (var i = 0; i < contentArr.length; i++) {
		if (contentArr[i] instanceof Folder) 	{	// is a folder: recall us
			walkFolder(contentArr[i]);
		}
		else {	// is a file
			if (contentArr[i].name.indexOf(".") == 0) continue;	// is hidden Unix file
			// check if it is a QuarkXpress file
			if (isXPressFile(contentArr[i]) == true) {
				// convert it
				convertDocument(contentArr[i], contentArr[i].path);
			}
		}
	}
}


/****************************
 * Open a document, call pages export and close it
 */
function convertDocument(theFileToConvert, theExportPath) {
	// prepare the INDD name
	var f = new File(theFileToConvert.fsName + ".indd");
	if (overwriteExistingINDD == false) {	// check if exists
		if (f.exists == true) return;
	}

	var myDocument = null;
	var myDocumentWin = null;
	$.gc();
	try {
		myDocument = app.open(theFileToConvert,false);	// do not show window now! this prevents some dialogs from beeing shown!
	}
	catch (Exception) { return; } 
	finally {
		if (myDocument == null) return;
		myDocumentWin = myDocument.windows.add();		// show the document window ...
		var bounds = new Array(100,100,900,750);		// ... and size it
		myDocumentWin.bounds = bounds;
		$.gc();
		$.sleep(3000);
	}
	// save the document as indd
	myDocument = app.activeDocument.save(f, false);

	// close it
	app.activeDocument.close();
	// inc doc counter
	numDocsConverted++;
	$.gc();
	$.sleep(1000);
}




/****************************
 * check if file is a QuarkXpress file
 */
function isXPressFile(thefile) {
	if (thefile == null) return(false);
	var err = thefile.open("r");
	thefile.encoding = "BINARY"; 
	err = thefile.seek(2,0);				// skip two zero bytes
	var content = "" + thefile.read(15);	// try to read 'MMXPR'
	err = thefile.close();

	// is QXP file?
	if (content.indexOf("MMXPR") >= 0) return(true);
	return(false);
}
